import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/auth";

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireSession(req, res);
  if (!session) return;
  if (req.method !== "GET")
    return res.status(405).json({ ok: false, error: "Method not allowed" });

  const role = (session.user as any)?.role;
  const userId = (session.user as any)?.id;

  // Parse query parameters for filters
  const { status, severity, department, search } = req.query;

  let where: any = { deletedAt: null };
  if (role === "ADMIN") {
    // no extra filter - admins see all tickets
  } else if (role === "TECHNICIAN") {
    // technicians see tickets assigned to them OR tickets they created
    where = {
      AND: [
        where,
        {
          OR: [{ assignedToId: userId }, { createdById: userId }],
        },
      ],
    };
  } else {
    // normal users see only tickets they created
    where = { AND: [where, { createdById: userId }] };
  }

  // Apply filters
  if (status) {
    if (status === "open") {
      where.closedAt = null;
    } else if (status === "closed") {
      where.closedAt = { not: null };
    }
  }

  if (severity) {
    where.severity = parseInt(severity as string);
  }

  if (department) {
    where.department = department;
  }

  if (search) {
    where.OR = [
      { title: { contains: search as string, mode: "insensitive" } },
      { description: { contains: search as string, mode: "insensitive" } },
    ];
  }

  const tickets = await prisma.ticket.findMany({
    where,
    orderBy: { createdAt: "desc" },
    select: {
      id: true,
      title: true,
      description: true,
      location: true,
      severity: true,
      department: true,
      txHash: true,
      chainTicketId: true,
      createdById: true,
      assignedToId: true,
      closedAt: true,
      lastKnownStatus: true,
      createdAt: true,
      anonymousEmail: true,
      anonymousName: true,
      createdBy: {
        select: {
          id: true,
          email: true,
          name: true,
          role: true,
        },
      },
      assignedTo: {
        select: {
          id: true,
          email: true,
          name: true,
          role: true,
        },
      },
    },
  });

  res.status(200).json({ ok: true, tickets });
}
